"use client";

import Link from "next/link";
import { FaLinkedin } from "react-icons/fa";
import Logo from "./Logo";
import { useEffect, useRef } from "react";

export default function Footer() {
  const videoRef = useRef<HTMLVideoElement>(null);

  useEffect(() => {
    const video = videoRef.current;
    if (!video) return;

    const observer = new IntersectionObserver(
      (entries) => {
        entries.forEach((entry) => {
          if (entry.isIntersecting) {
            // Load and play video when footer is visible
            if (!video.src) {
              video.src = "/video/footer.mp4";
              video.load();
            }
            video.play().catch(() => {
              // Handle autoplay failure silently
            });
          } else {
            // Pause video when footer is not visible
            video.pause();
          }
        });
      },
      { threshold: 0.1 }
    );

    observer.observe(video);

    return () => {
      observer.disconnect();
    };
  }, []);

  return (
    <footer
      className="overflow-hidden bg-[rgb(var(--main))] text-white px-6 md:px-15 lg:px-20 pt-28 pb-16 sticky bottom-0 w-full z-0"
      role="contentinfo"
      aria-label="Site footer"
    >
      {/* Background Video */}
      <div
        className="absolute z-0 w-full top-0 h-full left-0"
        aria-hidden="true"
      >
        <video
          ref={videoRef}
          muted
          loop
          preload="none"
          playsInline
          className="w-full h-full object-cover opacity-40 object-[50%_40%] md:object-[50%_60%] lg:object-[50%_70%] "
          aria-label="Background ocean video"
        />
        <div className="absolute bg-linear-to-b from-black/30 via-black/20 to-black/40 inset-0" />
      </div>

      {/* Content */}
      <div className="flex flex-col items-start relative z-10 max-w-screen-2xl mx-auto">
        {/* Logo */}
        <div className="pb-2 md:pb-0">
          <Link
            href="/"
            className="inline-block group"
            aria-label="Ocean of Olives homepage"
          >
            <Logo showText={true} className="w-20 md:w-24 h-fit" />
          </Link>
        </div>

        {/* Mobile Divider */}
        <div
          className="block md:hidden bg-white/30 h-px w-full shadow-sm"
          aria-hidden="true"
        />

        {/* Navigation & Social Icons */}
        <div className="flex flex-col md:flex-row md:justify-between w-full items-start md:items-center gap-8 md:gap-0">
          {/* Navigation Links */}
          <nav
            className="flex flex-col md:flex-row text-[1rem] py-6 gap-6.5 text-white/90"
            aria-label="Footer navigation"
          >
            <Link href="/" className="hover:text-white duration-100">
              Home
            </Link>
            <Link href="/about" className="hover:text-white duration-100">
              About
            </Link>
            <a
              href="mailto:support@oceanofolives.com"
              className="hover:text-white duration-100"
              aria-label="Contact us via email"
            >
              Contact us
            </a>
          </nav>
          {/* Desktop Social Icons */}
          {/* <div
            className="hidden md:flex gap-6 mt-4 md:mt-0"
            role="navigation"
            aria-label="Social media links"
          >
            <a
              rel="noopener noreferrer"
              target="_blank"
              href="https://www.linkedin.com/company/Oceanai/"
              aria-label="Visit Ocean of Olives on LinkedIn"
            >
              <FaLinkedin
                className="w-5 h-5 text-white drop-shadow-lg"
                aria-hidden="true"
              />
            </a>
          </div> */}
        </div>

        <div className="bg-white/20 h-px w-full" aria-hidden="true" />

        {/* mobile Social Icons */}
        {/* <div
          className="flex md:hidden gap-6 mt-4 md:mt-0"
          role="navigation"
          aria-label="Social media links"
        >
          <a
            rel="noopener noreferrer"
            target="_blank"
            href="https://www.linkedin.com/company/Oceanai/"
            aria-label="Visit Ocean of Olives on LinkedIn"
          >
            <FaLinkedin
              className="w-10 h-10 text-white drop-shadow-lg"
              aria-hidden="true"
            />
          </a>
        </div> */}

        {/* Copyright & Footer Links */}
        <div className="mt-10 md:mt-8 opacity-60 md:opacity-80 flex flex-col-reverse justify-between sm:flex-row w-full gap-4">
          <p className="text-[0.6rem] tracking-wide">
            Copyright © {new Date().getFullYear()} Ocean of Olives AI. All
            rights reserved.
          </p>
          <nav className="text-[0.6rem] flex gap-6" aria-label="Legal links">
            <Link
              href="/policies"
              className="hover:opacity-100 transition-all duration-300"
            >
              Privacy Policy
            </Link>
            <Link
              href="/terms"
              className="hover:opacity-100 transition-all duration-300"
            >
              Terms of Service
            </Link>
          </nav>
        </div>
      </div>
    </footer>
  );
}
